/*
 GUI for Simple BGC (Brushless Gimbal Controller) by Aleksey Moskalenko
 http://www.simplebgc.com
 (Originaly based on GUI for MultiWii Flight Controller: http://www.multiwii.com)

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version. see <http://www.gnu.org/licenses/>
*/
import processing.serial.*; // serial library
import controlP5.*; // controlP5 library
import processing.opengl.*;

Serial g_serial;
ControlP5 controlP5;
Textlabel txtlblWhichcom; 
ListBox commListbox;

public static final int COM_SPEED = 115200;
public static final int FRAMERATE = 20;
//public static final int COM_SPEED = 57600;
//public static final int FRAMERATE = 15;

/* Axis constants */
public static int NUM_AXIS = 2;
public static final int ROLL = 0, PITCH = 1, YAW = 2;
public static final String AXIS_NAME[] = { "ROLL", "PITCH", "YAW" };
public static final int TOGGLE_SIZE = 12;
public static final int DEBUG_VARS_NUM = 5;


int commListMax;
int serialState = 0;

cGraph g_graph;
int windowsX    = 670;        int windowsY    = 540;
int xGraph      = 10;         int yGraph      = 325;
int xObj        = 550;        int yObj        = 350;
int xRC         = 490;        int yRC         = 10;
int xParam      = 120;        int yParam      = 10;
int xRange      = 120;        int yRange      = 100;
int xExtGain    = 120;        int yExtGain    = 200;
int xLPF        = 320;        int yLPF        = 200;


/****************** Error definition ****************/
class ErrorInfo {
	int code;
	public final String text;
	public final String hint;
	
	ErrorInfo(int code, String text, String hint) {
		this.code = code;
		this.text = text;
		this.hint = hint;
	}
}

// Frmware errors
public static final int ERR_NO_SENSOR = 1;
public static final int ERR_CALIB_ACC = 2;
public static final int ERR_SET_POWER = 4;
public static final int ERR_CALIB_POLES = 8;
// GUI errors
public static final int ERR_GUI_VERSION = 128;


private ErrorInfo errors[] = {
	new ErrorInfo(ERR_NO_SENSOR, "Sensor is not connected", "Turn off power and check I2C sensor connection. DON'T CONNECT SENSOR WHEN BOARD IS POWERED!"),
	new ErrorInfo(ERR_CALIB_ACC, "Accelerometer is not calibrated", "Simple calibration: Level sensor horisontally as precise as possible, and than press \"CALIB_ACC\". For more precise six-points calibration see instruction manual"),
	new ErrorInfo(ERR_SET_POWER, "POWER is not set",  "Increse POWER from low values (50 for example) until motor will give enough torque, but does not make too hot"),
	new ErrorInfo(ERR_CALIB_POLES, "POLES is not set",  "Use AUTO calibration to get number of poles and direction. Set precise number of poles manually if detected wrong (it equals to number of magnets in the motor)"),
	new ErrorInfo(ERR_GUI_VERSION, "GUI vs board version mismatch",  "Try to update GUI to latest version")
	
};

public ErrorInfo getErrorInfo(int code) {
	if(code > 0) {
		for(int i=0; i<errors.length; i++) {
			if((errors[i].code & code) > 0) {
				return errors[i];
			}
		}
		return new ErrorInfo(0, "UNKNOWN ERROR #" + code, "SEE MANUALS TO CHECK CODE");
	} else {
		return null;
	}
}

ErrorInfo error = null;

/**************************************************/


boolean axGraph =true,ayGraph=true,azGraph=true,gxGraph=true,gyGraph=true,gzGraph=true, magxGraph =false,magyGraph=false,magzGraph=false;
boolean debug1Graph = false,debug2Graph = false,debug3Graph = false,debug4Graph = false;

/* Graph data */
cDataArray accPITCH   = new cDataArray(100), accROLL    = new cDataArray(100), accYAW     = new cDataArray(100);
cDataArray gyroPITCH  = new cDataArray(100), gyroROLL   = new cDataArray(100), gyroYAW    = new cDataArray(100);
cDataArray magxData   = new cDataArray(100), magyData   = new cDataArray(100), magzData   = new cDataArray(100);
cDataArray headData    = new cDataArray(100);
cDataArray debug1Data   = new cDataArray(100),debug2Data   = new cDataArray(100),debug3Data   = new cDataArray(100),debug4Data   = new cDataArray(100);



/* Config edit cotrols */
Numberbox confP[] = new Numberbox[3], confI[] = new Numberbox[3], confD[] = new Numberbox[3];
Toggle confInv[] = new Toggle[3];
Numberbox confPower[] = new Numberbox[3];
Numberbox confPoles[] = new Numberbox[3];
Numberbox confRcMinAngle[] = new Numberbox[3], confRcMaxAngle[] = new Numberbox[3];
Numberbox confRcLPF[] = new Numberbox[3];
Toggle confRcInc[] = new Toggle[3];
Numberbox confExtGain[] = new Numberbox[2];
Numberbox confGyroLPF;


Numberbox confDebug[] = new Numberbox[DEBUG_VARS_NUM];
Button	buttonSAVE_DEBUG;



/* State and debug view controls */
Slider rcStickRollSlider,rcStickPitchSlider,rcStickYawSlider;
Slider extRollSlider,extPitchSlider;

Slider axSlider,aySlider,azSlider,gxSlider,gySlider,gzSlider , magxSlider,magySlider,magzSlider , baroSlider,headSlider;
Slider debug1Slider,debug2Slider,debug3Slider,debug4Slider;

Slider scaleSlider;

Button buttonREAD,
	buttonWRITE,
	buttonCALIBRATE_ACC,
	buttonUSE_DEFAULTS,
	//buttonSTART,
	//buttonSTOP,
	buttonAUTO_GAIN,
	buttonAUTO_POLES;


color green_ = color(30, 120, 30), gray_ = color(60, 60, 60), red_ = color(255,0,0), blue_ = color(30, 90, 120);
boolean graphEnable = false;boolean readEnable = false;boolean writeEnable = false;boolean calibrateEnable = false;

int version,subversion;
float head,angx,angy;
int init_com; //,graph_on;

short cycleCnt = 0;

int cycleTime,i2cError;

PFont font8,font12,font15;


// Angle meter
final int ANG_METER_SCALE = 10;
PeakMeter peakRoll = new PeakMeter(20);
PeakMeter peakPitch = new PeakMeter(20);
AvgFilter avgRoll = new AvgFilter(20);
AvgFilter avgPitch = new AvgFilter(20);



// coded by Eberhard Rensch
// Truncates a long port name for better (readable) display in the GUI
String shortifyPortName(String portName, int maxlen)  {
  String shortName = portName;
  if(shortName.startsWith("/dev/")) shortName = shortName.substring(5);  
  if(shortName.startsWith("tty.")) shortName = shortName.substring(4); // get rid off leading tty. part of device name
  if(portName.length()>maxlen) shortName = shortName.substring(0,(maxlen-1)/2) + "~" +shortName.substring(shortName.length()-(maxlen-(maxlen-1)/2));
  if(shortName.startsWith("cu.")) shortName = "";// only collect the corresponding tty. devices
  return shortName;
}

controlP5.Controller hideLabel(controlP5.Controller c) {
  c.setLabel("");
  c.setLabelVisible(false);
  return c;
}

void setup() {
  size(windowsX,windowsY,OPENGL);
  frameRate(FRAMERATE); 

  font8 = createFont("Arial bold",8,false);font12 = createFont("Arial bold",12,false);font15 = createFont("Arial bold",15,false);

  controlP5 = new ControlP5(this); // initialize the GUI controls
  controlP5.setControlFont(font12);
  
  g_graph  = new cGraph(xGraph+110,yGraph, 480, 200);
  commListbox = controlP5.addListBox("portComList",5,75,110,240); // make a listbox and populate it with the available comm ports

  commListbox.captionLabel().set("PORT COM");
  commListbox.setColorBackground(gray_);
  for(int i=0;i<Serial.list().length;i++) {
    String pn = shortifyPortName(Serial.list()[i], 13);
    if (pn.length() >0 ) commListbox.addItem(pn,i); // addItem(name,value)
    commListMax = i;
  }
  commListbox.addItem("Close Comm",++commListMax); // addItem(name,value)
  // text label for which comm port selected
  txtlblWhichcom = controlP5.addTextlabel("txtlblWhichcom","No Port Selected",5,50); // textlabel(name,text,x,y)
    
  //buttonSTART = controlP5.addButton("bSTART",1,xGraph+110,yGraph-25,40,19); buttonSTART.setLabel("START"); buttonSTART.setColorBackground(gray_);
  //buttonSTOP = controlP5.addButton("bSTOP",1,xGraph+160,yGraph-25,40,19); buttonSTOP.setLabel("STOP"); buttonSTOP.setColorBackground(gray_);

  color c,black;
  black = color(0,0,0);
  int xo = xGraph-7;
  int x = xGraph+40;
  int y1= yGraph+10;  //ACC
  int y2= yGraph+55;  //GYRO
  int y5= yGraph+100; //MAG
  int y6= yGraph+145; //DEBUG

  
  /******* Debug controls ******/
  Toggle tAX =     controlP5.addToggle("ACC_ROLL",true,x,y1+10,20,10);tAX.setColorActive(color(255, 0, 0));tAX.setColorBackground(black);tAX.setLabel(""); tAX.setValue(axGraph);
  Toggle tAY =   controlP5.addToggle("ACC_PITCH",true,x,y1+20,20,10);tAY.setColorActive(color(0, 255, 0));tAY.setColorBackground(black);tAY.setLabel(""); tAY.setValue(ayGraph);
  Toggle tAZ =           controlP5.addToggle("ACC_Z",true,x,y1+30,20,10);tAZ.setColorActive(color(0, 0, 255));tAZ.setColorBackground(black);tAZ.setLabel(""); tAZ.setValue(azGraph);
  Toggle tGX =   controlP5.addToggle("GYRO_ROLL",true,x,y2+10,20,10);tGX.setColorActive(color(200, 200, 0));tGX.setColorBackground(black);tGX.setLabel(""); tGX.setValue(gxGraph);
  Toggle tGY = controlP5.addToggle("GYRO_PITCH",true,x,y2+20,20,10);tGY.setColorActive(color(0, 255, 255));tGY.setColorBackground(black);tGY.setLabel("");  tGY.setValue(gyGraph);
  Toggle tGZ =     controlP5.addToggle("GYRO_YAW",true,x,y2+30,20,10);tGZ.setColorActive(color(255, 0, 255));tGZ.setColorBackground(black);tGZ.setLabel(""); tGZ.setValue(gzGraph);
  Toggle tMAGX =             controlP5.addToggle("MAGX",true,x,y5+10,20,10);tMAGX.setColorActive(color(50, 100, 150));tMAGX.setColorBackground(black);tMAGX.setLabel("");  tMAGX.setValue(magxGraph);
  Toggle tMAGY =             controlP5.addToggle("MAGY",true,x,y5+20,20,10);tMAGY.setColorActive(color(100, 50, 150));tMAGY.setColorBackground(black);tMAGY.setLabel(""); tMAGY.setValue(magyGraph);
  Toggle tMAGZ =             controlP5.addToggle("MAGZ",true,x,y5+30,20,10);tMAGZ.setColorActive(color(150, 100, 50));tMAGZ.setColorBackground(black);tMAGZ.setLabel(""); tMAGZ.setValue(magzGraph);
  Toggle tDEBUG1 =         controlP5.addToggle("DEBUG1",true,x,y6+10,20,10);tDEBUG1.setColorActive(color(50, 100, 150));tDEBUG1.setColorBackground(black);tDEBUG1.setLabel("");tDEBUG1.setValue(debug1Graph);tDEBUG1.hide();
  Toggle tDEBUG2 =         controlP5.addToggle("DEBUG2",true,x,y6+20,20,10);tDEBUG2.setColorActive(color(100, 50, 150));tDEBUG2.setColorBackground(black);tDEBUG2.setLabel("");tDEBUG2.setValue(debug2Graph);
  Toggle tDEBUG3 =         controlP5.addToggle("DEBUG3",true,x,y6+30,20,10);tDEBUG3.setColorActive(color(150, 100, 50));tDEBUG3.setColorBackground(black);tDEBUG3.setLabel("");tDEBUG3.setValue(debug3Graph);
  Toggle tDEBUG4 =         controlP5.addToggle("DEBUG4",true,x,y6+40,20,10);tDEBUG4.setColorActive(color(150, 150, 0));tDEBUG4.setColorBackground(black);tDEBUG4.setLabel("");tDEBUG4.setValue(debug4Graph);

  controlP5.addTextlabel("acclabel","ACC",xo,y1);
  controlP5.addTextlabel("accrolllabel","   ROLL",xo,y1+10);
  controlP5.addTextlabel("accpitchlabel","   PITCH",xo,y1+20);
  controlP5.addTextlabel("acczlabel","   Z",xo,y1+30);
  controlP5.addTextlabel("gyrolabel","GYRO",xo,y2);
  controlP5.addTextlabel("gyrorolllabel","   ROLL",xo,y2+10);
  controlP5.addTextlabel("gyropitchlabel","   PITCH",xo,y2+20);
  controlP5.addTextlabel("gyroyawlabel","   YAW",xo,y2+30);
  controlP5.addTextlabel("maglabel","MAG",xo,y5);
  controlP5.addTextlabel("magrolllabel","   ROLL",xo,y5+10);
  controlP5.addTextlabel("magpitchlabel","   PITCH",xo,y5+20);
  controlP5.addTextlabel("magyawlabel","   YAW",xo,y5+30);
  controlP5.addTextlabel("debug1","debug1",xo,y6+10);
  controlP5.addTextlabel("debug2","debug2",xo,y6+20);
  controlP5.addTextlabel("debug3","debug3",xo,y6+30);
  controlP5.addTextlabel("debug4","debug4",xo,y6+40);

  axSlider   =       controlP5.addSlider("axSlider",-1000,+1000,0,x+20,y1+10,50,10);axSlider.setDecimalPrecision(0);axSlider.setLabel("");
  aySlider   =       controlP5.addSlider("aySlider",-1000,+1000,0,x+20,y1+20,50,10);aySlider.setDecimalPrecision(0);aySlider.setLabel("");
  azSlider   =       controlP5.addSlider("azSlider",-1000,+1000,0,x+20,y1+30,50,10);azSlider.setDecimalPrecision(0);azSlider.setLabel("");
  gxSlider   =       controlP5.addSlider("gxSlider",-5000,+5000,0,x+20,y2+10,50,10);gxSlider.setDecimalPrecision(0);gxSlider.setLabel("");
  gySlider   =       controlP5.addSlider("gySlider",-5000,+5000,0,x+20,y2+20,50,10);gySlider.setDecimalPrecision(0);gySlider.setLabel("");
  gzSlider   =       controlP5.addSlider("gzSlider",-5000,+5000,0,x+20,y2+30,50,10);gzSlider.setDecimalPrecision(0);gzSlider.setLabel("");
  magxSlider  =      controlP5.addSlider("magxSlider",-5000,+5000,0,x+20,y5+10,50,10);magxSlider.setDecimalPrecision(0);magxSlider.setLabel("");
  magySlider  =      controlP5.addSlider("magySlider",-5000,+5000,0,x+20,y5+20,50,10);magySlider.setDecimalPrecision(0);magySlider.setLabel("");
  magzSlider  =      controlP5.addSlider("magzSlider",-5000,+5000,0,x+20,y5+30,50,10);magzSlider.setDecimalPrecision(0);magzSlider.setLabel("");
  debug1Slider  =    controlP5.addSlider("debug1Slider",-32000,+32000,0,x+20,y6+10,50,10);debug1Slider.setDecimalPrecision(0);debug1Slider.setLabel("");
  debug2Slider  =    controlP5.addSlider("debug2Slider",-32000,+32000,0,x+20,y6+20,50,10);debug2Slider.setDecimalPrecision(0);debug2Slider.setLabel("");
  debug3Slider  =    controlP5.addSlider("debug3Slider",-32000,+32000,0,x+20,y6+30,50,10);debug3Slider.setDecimalPrecision(0);debug3Slider.setLabel("");
  debug4Slider  =    controlP5.addSlider("debug4Slider",-32000,+32000,0,x+20,y6+40,50,10);debug4Slider.setDecimalPrecision(0);debug4Slider.setLabel("");

  scaleSlider = controlP5.addSlider("SCALE",0,10,1,xGraph+515,yGraph,75,20);scaleSlider.setLabel("");



	/******** PID controls **********************/
  // Draw PID labels
  controlP5.addTextlabel("confP", "P", xParam+45, yParam+5);
  controlP5.addTextlabel("confI", "I", xParam+90, yParam+5);
  controlP5.addTextlabel("confD", "D", xParam+130, yParam+5);
  controlP5.addTextlabel("confPower", "POWER", xParam+155, yParam+5);
  controlP5.addTextlabel("confInvert", "INVERT", xParam+210, yParam+5);
  controlP5.addTextlabel("confPoles", "N.POLES", xParam+260, yParam+5);
  for(int i=0;i<3;i++) {
    controlP5.addTextlabel("confLabel"+i, AXIS_NAME[i], xParam+3, yParam + 23 + i*20);
    confP[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confP"+i,0,xParam+40,yParam+20+i*20,30,14));
    confP[i].setColorBackground(gray_);confP[i].setMin(0);confP[i].setDirection(Controller.HORIZONTAL);confP[i].setDecimalPrecision(0);confP[i].setMultiplier(1);confP[i].setMax(50);
    confI[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confI"+i,0,xParam+75,yParam+20+i*20,40,14));
    confI[i].setColorBackground(gray_);confI[i].setMin(0);confI[i].setDirection(Controller.HORIZONTAL);confI[i].setDecimalPrecision(2);confI[i].setMultiplier(0.01);confI[i].setMax(0.50);
    confD[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confD"+i,0,xParam+120,yParam+20+i*20,30,14));
    confD[i].setColorBackground(gray_);confD[i].setMin(0);confD[i].setDirection(Controller.HORIZONTAL);confD[i].setDecimalPrecision(0);confD[i].setMultiplier(1);confD[i].setMax(50);

    confPower[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confPower"+i,0,xParam+160,yParam+20+i*20,30,14));
    confPower[i].setColorBackground(gray_);confPower[i].setMin(0);confPower[i].setDirection(Controller.HORIZONTAL);confPower[i].setDecimalPrecision(0);confPower[i].setMultiplier(1);confPower[i].setMax(255);

    confInv[i] =  controlP5.addToggle("confInv"+i, false, xParam+230,yParam+22+i*20, TOGGLE_SIZE, TOGGLE_SIZE);
    confInv[i].setColorActive(color(255));confInv[i].setColorBackground(color(120));confInv[i].setLabelVisible(false);
    //TODO hideLabel(confInv[i].addItem(i + "1", 1));

    confPoles[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confPoles"+i,0,xParam+260,yParam+20+i*20,30,14));
    confPoles[i].setColorBackground(gray_);confPoles[i].setMin(0);confPoles[i].setDirection(Controller.HORIZONTAL);confPoles[i].setDecimalPrecision(0);confPoles[i].setMultiplier(1);confPoles[i].setMax(255);
  }
  
  /********** RC controls *********************/
  controlP5.addTextlabel("rangeFrom", "RC min.angle", xRange + 40, yRange);
  controlP5.addTextlabel("rangeTo", "RC max.angle", xRange + 130, yRange);
  controlP5.addTextlabel("RcLPF", "LPF/SPD", xRange + 220, yRange);
  controlP5.addTextlabel("RcInc", "INC.MODE", xRange + 285, yRange);
  for(int i=0;i<3;i++) {
    controlP5.addTextlabel("rangeLabel"+i, AXIS_NAME[i], xRange+3, yRange + 23 + i*20);
    confRcMinAngle[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confRcMinAngle"+i,0,xRange+60,yRange+20+i*20,40,14));
    confRcMinAngle[i].setColorBackground(gray_);confRcMinAngle[i].setMin(-180);confRcMinAngle[i].setDirection(Controller.HORIZONTAL);confRcMinAngle[i].setDecimalPrecision(0);confRcMinAngle[i].setMultiplier(1);confRcMinAngle[i].setMax(180);
    confRcMaxAngle[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confRcMaxAngle"+i,0,xRange+140,yRange+20+i*20,40,14));
    confRcMaxAngle[i].setColorBackground(gray_);confRcMaxAngle[i].setMin(-180);confRcMaxAngle[i].setDirection(Controller.HORIZONTAL);confRcMaxAngle[i].setDecimalPrecision(0);confRcMaxAngle[i].setMultiplier(1);confRcMaxAngle[i].setMax(180);
    confRcLPF[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confRcLPF"+i,0,xRange+220,yRange+20+i*20,30,14));
    confRcLPF[i].setColorBackground(gray_);confRcLPF[i].setMin(0);confRcLPF[i].setDirection(Controller.HORIZONTAL);confRcLPF[i].setDecimalPrecision(0);confRcLPF[i].setMultiplier(1);confRcLPF[i].setMax(10);

  	confRcInc[i] = controlP5.addToggle("confRcInc"+i,false,xRange+300,yRange+20+i*20, TOGGLE_SIZE, TOGGLE_SIZE);
  	confRcInc[i].setColorActive(color(255));confRcInc[i].setColorBackground(color(120));confRcInc[i].setLabelVisible(false);
  }
  
  /********** Ext FC gain ******************/
  controlP5.addTextlabel("extGain", "EXTERNAL FC GAIN", xExtGain+40, yExtGain);
  for(int i=0;i<2;i++) {
    controlP5.addTextlabel("extGain"+i, AXIS_NAME[i], xExtGain+3, yExtGain + 23 + i*20);
    confExtGain[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confExtGain"+i,0,xExtGain+40,yExtGain+20+i*20,30,14));
    confExtGain[i].setColorBackground(gray_);confExtGain[i].setMin(-128);confExtGain[i].setDirection(Controller.HORIZONTAL);confExtGain[i].setDecimalPrecision(0);confExtGain[i].setMultiplier(1);confExtGain[i].setMax(127);
  }

  
  /********** LPF Filters *********************/
  // TODO: not implemented in firmware
  controlP5.addTextlabel("gyroLPF", "GYRO LPF:", xLPF, yLPF);
  confGyroLPF = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confGyroLPF",0,xLPF+80,yLPF-3,30,14));
  confGyroLPF.setColorBackground(gray_);confGyroLPF.setMin(0);confGyroLPF.setDirection(Controller.HORIZONTAL);confGyroLPF.setDecimalPrecision(0);confGyroLPF.setMultiplier(1);confGyroLPF.setMax(5);
  
  /********** Debug params *******************/
  int yVars = 145;
  controlP5.addTextlabel("debugVars", "DEBUG VARS:", 15, yVars);
  yVars+=20;
  for(int i=0;i<DEBUG_VARS_NUM;i++) {
	  controlP5.addTextlabel("labelVar"+i, "var"+i+":", 20, yVars+4);
    confDebug[i] = (controlP5.Numberbox) hideLabel(controlP5.addNumberbox("confDebug"+i,0,55,yVars,40,14));
    confDebug[i].setMin(-32000);confDebug[i].setDirection(Controller.HORIZONTAL);confDebug[i].setDecimalPrecision(0);confDebug[i].setMultiplier(1);confDebug[i].setMax(32000);
    yVars+= 20;
	}	  
  buttonSAVE_DEBUG =  controlP5.addButton("SAVE_DEBUG",1,20,yVars+5,70,16); buttonSAVE_DEBUG.setLabel("    SAVE");

  
  
  
  /****** Buttons ***********/
  buttonREAD =      controlP5.addButton("READ",1,xParam+5,yParam+260,60,16);buttonREAD.setColorBackground(gray_);
  buttonWRITE =     controlP5.addButton("WRITE",1,xParam+290,yParam+260,60,16);buttonWRITE.setColorBackground(gray_);
  buttonCALIBRATE_ACC = controlP5.addButton("CALIB_ACC",1,xParam+210,yParam+260,70,16);buttonCALIBRATE_ACC.setColorBackground(gray_);
  buttonUSE_DEFAULTS = controlP5.addButton("USE_DEFAULTS",1,xParam+75,yParam+260,100,16);buttonUSE_DEFAULTS.setColorBackground(gray_);
  buttonAUTO_GAIN =      controlP5.addButton("CALIB_EXT_GAIN",1,xExtGain+80,yExtGain+20,80,35); buttonAUTO_GAIN.setLabel("     AUTO");buttonAUTO_GAIN.setColorBackground(gray_);
  buttonAUTO_POLES =      controlP5.addButton("CALIB_POLES",1,xParam+295,yParam+20,50,35); buttonAUTO_POLES.setLabel("  AUTO");buttonAUTO_POLES.setColorBackground(gray_);

  /********* RC input sliders *********/
  controlP5.addTextlabel("rcLabel", "RC INPUT", xRC+5, yRC+5);
  rcStickRollSlider =     controlP5.addSlider("rcRoll",900,2100,1500,xRC+5,yRC+20,100,10);rcStickRollSlider.setDecimalPrecision(0);
  rcStickPitchSlider =    controlP5.addSlider("rcPitch",900,2100,1500,xRC+5,yRC+35,100,10);rcStickPitchSlider.setDecimalPrecision(0);
  rcStickYawSlider  =     controlP5.addSlider("rcYaw",900,2100,1500,xRC+5,yRC+50,100,10);rcStickYawSlider.setDecimalPrecision(0);

  /********* Ext FC input sliders *********/
  controlP5.addTextlabel("extLabel", "EXT FC INPUT", xRC+5, yRC+70);
  extRollSlider =     controlP5.addSlider("extRoll",900,2100,1500,xRC+5,yRC+85,100,10); extRollSlider.setDecimalPrecision(0);
  extPitchSlider =    controlP5.addSlider("extPitch",900,2100,1500,xRC+5,yRC+100,100,10); extPitchSlider.setDecimalPrecision(0);
  
  showControls(DEBUG_CONTROLS, false);
  showControls(YAW_CONTROLS, false);
}

void draw() {
  int i;
  float val,inter,a,b,h;
 
  cycleCnt++;
  
  background(80);
  textFont(font12); fill(0xCC, 0xCC, 0xCC);
  text("Board ver:",5,20); text(version/10 + "." + version%10 + (subversion > 0 ? " b" + subversion : ""), 70, 20);

  textFont(font15); 
  if(i2cError > 100 && timeToFlash()) {
  	fill(255,0,0);
  } else {
	  fill(255,255,255);
	}
  text(i2cError,xGraph+410,yGraph-10);
  fill(255,255,255);
  textFont(font12);
  text("I2C error:",xGraph+350,yGraph-10);
  
  if(showDebug) {
	  textFont(font12);
	  text("Cycle Time:",xGraph+220,yGraph-10);
	  textFont(font15); 
  	text(cycleTime,xGraph+290,yGraph-10);
  }


  /* Process serial data */
  if (init_com==1) {
  	processSerial();
 		if(graphEnable) {
    	g_serial.write('D');
    }
  }


  stroke(255); 
  
  // Draw orientation?
  if (angx<-90) a=radians(-180 - angx);
  else if (angx>90) a=radians(+180 - angx);
  else a=radians(angx);
  b=radians(angy);
  h=radians(head);

  float size = 30.0;


  // Rotate?
  pushMatrix();
  translate(xObj+60,yObj-170);
  rotate(a);
  textFont(font15);text("ROLL", -20, 15);
  line(-30,0,+30,0);line(0,0,0,-10);
  popMatrix();
  
  pushMatrix();
  translate(xObj+60,yObj-90);
  rotate(b);
  textFont(font15);text("PITCH", -30, 15);
  line(-30,0,30,0);line(+30,0,30-size/3 ,size/3);line(+30,0,30-size/3 ,-size/3);  
  popMatrix();
 


  // Draw compass?
  pushMatrix();
  translate(xObj-20,yObj-133);
  size=15;
  strokeWeight(1.5);
  fill(160);stroke(160);
  ellipse(0,  0,   4*size+7, 4*size+7);

  strokeWeight(4);stroke(200);line(0,0, 0,-3*size);line(0,-3*size, -5 ,-3*size+10); line(0,-3*size, +5 ,-3*size+10);  

  strokeWeight(1.5);fill(0);stroke(0);ellipse(0,  0,   2*size+7, 2*size+7);
  stroke(255);

  rotate(head*PI/180);
  line(0,size, 0,-size); line(0,-size, -5 ,-size+10); line(0,-size, +5 ,-size+10);
  popMatrix();
  text("N",xObj-25,yObj-155);text("S",xObj-25,yObj-100);
  text("W",xObj-53,yObj-127);text("E",xObj   ,yObj-127);

  // Draw curent delta angle
  stroke(110); fill(110);
  rect(xObj+60, yObj-120, xObj+60+constrain((int)((angx - avgRoll.res())*ANG_METER_SCALE), -50, 50), yObj-110);
  rect(xObj+60, yObj-40, xObj+60+constrain((int)((angy - avgPitch.res())*ANG_METER_SCALE), -50, 50), yObj-30);

  // Draw peak delta angle
  stroke(255,0,0);
  int tmp = xObj+60 + constrain((int)(peakRoll.getMin()*ANG_METER_SCALE), -50, 0);
  line(tmp, yObj-120, tmp, yObj-110); 
  tmp = xObj+60 + constrain((int)(peakRoll.getMax()*ANG_METER_SCALE), 0, 50);
  line(tmp, yObj-120, tmp, yObj-110); 
  tmp = xObj+60 + constrain((int)(peakPitch.getMin()*ANG_METER_SCALE), -50, 0);
  line(tmp, yObj-40, tmp, yObj-30); 
  tmp = xObj+60 + constrain((int)(peakPitch.getMax()*ANG_METER_SCALE), 0, 50);
  line(tmp, yObj-40, tmp, yObj-30); 
  textFont(font12);  fill(255,0,0);
	text(String.format("%.1f", peakRoll.getMax() - peakRoll.getMin()), xObj+80, yObj-130);
	text(String.format("%.1f", peakPitch.getMax() - peakPitch.getMin()), xObj+80, yObj-50);

	// Draw axis angles and peak meter
  textFont(font12);  fill(160);
	text(String.format("%.1f", angx), xObj+50, yObj-130);
	text(String.format("%.1f", angy), xObj+50, yObj-50);
	text(String.format("%.1f", head), xObj-30, yObj-80);
  

  
  // Draw graph
  strokeWeight(1);
  fill(255, 255, 255);
  g_graph.drawGraphBox();
  
  strokeWeight(1.5);
  stroke(255, 0, 0); if (axGraph) g_graph.drawLine(accROLL, -1000, +1000);
  stroke(0, 255, 0); if (ayGraph) g_graph.drawLine(accPITCH, -1000, +1000);
  stroke(0, 0, 255);
  if (azGraph) {
    if (scaleSlider.value()<2) g_graph.drawLine(accYAW, -1000, +1000);
    else g_graph.drawLine(accYAW, 200*scaleSlider.value()-1000,200*scaleSlider.value()+500);
  }
  
  stroke(200, 200, 0);  if (gxGraph)   g_graph.drawLine(gyroROLL, -500, +500);
  stroke(0, 255, 255);  if (gyGraph)   g_graph.drawLine(gyroPITCH, -500, +500);
  stroke(255, 0, 255);  if (gzGraph)   g_graph.drawLine(gyroYAW, -500, +500);
  stroke(50, 100, 150); if (magxGraph) g_graph.drawLine(magxData, -500, +500);
  stroke(100, 50, 150); if (magyGraph) g_graph.drawLine(magyData, -500, +500);
  stroke(150, 100, 50); if (magzGraph) g_graph.drawLine(magzData, -500, +500);

  stroke(50,100,150); if (debug1Graph)  g_graph.drawLine(debug1Data, -5000, +5000);
  stroke(100,50,150); if (debug2Graph)  g_graph.drawLine(debug2Data, -5000, +5000);
  stroke(150,100,50); if (debug3Graph)  g_graph.drawLine(debug3Data, -5000, +5000);
  stroke(150,150,0); if (debug4Graph)  g_graph.drawLine(debug4Data, -5000, +5000);

  fill(0, 0, 0);

  // Draw frames
  strokeWeight(3);stroke(0);
  rectMode(CORNERS);
  rect(xParam,yParam, xParam+355, yParam+280);
  rect(xRC,yRC, xRC+170, yRC+120);

  stroke(50);
  rect(xParam+220,yParam+15, xParam+350, yParam+60);

  /*
  int xSens       = xParam + 80;
  int ySens       = yParam + 210;
  stroke(255);
  a=min(confRC_RATE.value(),1);
  b=confRC_EXPO.value();
  strokeWeight(1);
  line(xSens,ySens,xSens,ySens+40);
  line(xSens,ySens+40,xSens+70,ySens+40);
  strokeWeight(3);stroke(30,120,30);
  for(i=0;i<70;i++) {
    inter = 10*i;
    val = a*inter*(1-b+inter*inter*b/490000);
    point(xSens+i,ySens+(70-val/10)*4/7);
  }
  if (confRC_RATE.value()>1) { 
    stroke(220,100,100);
    ellipse(xSens+70, ySens, 7, 7);
  }
  */
  
  if (error != null) {
  	textFont(font15);fill(#ff0000);text(error.text,150,400, 450, 430); 
  	textFont(font12);fill(#000000);text(error.hint,150,440, 450, 500); 
  }
}

void ACC_ROLL(boolean theFlag) {axGraph = theFlag;}
void ACC_PITCH(boolean theFlag) {ayGraph = theFlag;}
void ACC_Z(boolean theFlag) {azGraph = theFlag;}
void GYRO_ROLL(boolean theFlag) {gxGraph = theFlag;}
void GYRO_PITCH(boolean theFlag) {gyGraph = theFlag;}
void GYRO_YAW(boolean theFlag) {gzGraph = theFlag;}
void MAGX(boolean theFlag) {magxGraph = theFlag;}
void MAGY(boolean theFlag) {magyGraph = theFlag;}
void MAGZ(boolean theFlag) {magzGraph = theFlag;}
void DEBUG1(boolean theFlag) {debug1Graph = theFlag;}
void DEBUG2(boolean theFlag) {debug2Graph = theFlag;}
void DEBUG3(boolean theFlag) {debug3Graph = theFlag;}
void DEBUG4(boolean theFlag) {debug4Graph = theFlag;}

public void controlEvent(ControlEvent theEvent) {
  if (theEvent.isGroup()) { 
  	if (theEvent.name()=="portComList") InitSerial(theEvent.group().value()); // initialize the serial port selected
  }
}

/*
public void bSTART() {
  if(graphEnable == false) return;
  graph_on=1;
  g_serial.clear();
}

public void bSTOP() {
  graph_on=0;
}
*/

public void READ() {
  if(readEnable == false) return;
  g_serial.write('R');
  g_serial.write('r');
}

public void SerialWrite(float val) {
	g_serial.write(char(round(val)));
}
public void SerialWrite16(float val) {
	int c = round(val);
	g_serial.write((byte)(c&0xff));
	g_serial.write((byte)((c>>8)&0xff));
}

public void WRITE() {
  if(writeEnable == false) return;

	g_serial.write('W'); // write from GUI to controller
	// PID data
	for(int i=0;i<3;i++) {
		SerialWrite(confP[i].value());  
		SerialWrite(confI[i].value()*100);  
		SerialWrite(confD[i].value());
		SerialWrite(confPower[i].value());
		SerialWrite(confInv[i].getState() ? 1 : 0); //arrayValue()[0]);
		SerialWrite(confPoles[i].value());
	}
	// RC data
	for(int i=0;i<3;i++) {
		SerialWrite16(confRcMinAngle[i].value());
		SerialWrite16(confRcMaxAngle[i].value());
		SerialWrite(confRcLPF[i].value());
		SerialWrite(confRcInc[i].getState() ? 1 : 0);
	}
	
	// Ext FC data
	for(int i=0;i<2;i++) {
		SerialWrite(confExtGain[i].value());
	}
	
	// Other tuning
	SerialWrite(confGyroLPF.value());
}

public void SAVE_DEBUG() {
  if(writeEnable == false) return;

	g_serial.write('w'); // write from GUI to controller
	for(int i=0;i<DEBUG_VARS_NUM;i++) {
		SerialWrite16(confDebug[i].value());  
	}
}

public void CALIB_ACC() {
  if(calibrateEnable == false) return;
  g_serial.write('A'); // acc Sensor calibration request
}
public void USE_DEFAULTS() {
  if(calibrateEnable == false) return;
  g_serial.write('F'); // mag Sensor calibration request
}
public void CALIB_EXT_GAIN() {
  if(calibrateEnable == false) return;
  g_serial.write('G'); 
}
public void CALIB_POLES() {
  if(calibrateEnable == false) return;
  g_serial.write('P'); 
}

// initialize the serial port selected in the listBox
void InitSerial(float portValue) {
  if (portValue < commListMax) {
    String portPos = Serial.list()[int(portValue)];
    txtlblWhichcom.setValue("COM = " + shortifyPortName(portPos, 8));
    g_serial = new Serial(this, portPos, COM_SPEED);
    init_com=1;
    commListbox.setColorBackground(green_);
    //buttonSTART.setColorBackground(green_);
    //buttonSTOP.setColorBackground(green_);
	  buttonREAD.setColorBackground(green_); 
	  buttonCALIBRATE_ACC.setColorBackground(green_); 
	  buttonUSE_DEFAULTS.setColorBackground(green_);
	  buttonAUTO_GAIN.setColorBackground(blue_); 
	  buttonAUTO_POLES.setColorBackground(blue_); 
	  readEnable = true;calibrateEnable = true; graphEnable = true;
    g_serial.buffer(256);
    READ();
  } else {
    txtlblWhichcom.setValue("Comm Closed");
    init_com=0;
    //buttonSTART.setColorBackground(gray_);
    //buttonSTOP.setColorBackground(gray_);
    commListbox.setColorBackground(gray_);
    graphEnable = false;
    readEnable = false;
    init_com=0;
    g_serial.stop();
  }
}

/******* Serial reading buffer ********/
int p;
byte[] inBuf = new byte[512];
int read16() {return (inBuf[p++]&0xff) + (inBuf[p++]<<8);}
int read8()  {return inBuf[p++]&0xff;}

int serialReaded = 0;

/* Read up to 'size' bytes into buffer from serial, if available.
Return true on complete */
boolean serialReadBuf(int size) {
  while(g_serial.available() > 0 && serialReaded < size) {
    inBuf[serialReaded++] = (byte)g_serial.read();
  }
  return (serialReaded >= size);
}



/* Serial data reading */
void processSerial() { 
  while(g_serial.available()>0) {
		
		/**** Read command *****/
		if(serialState == 0) {
			serialState = g_serial.read();
	    serialReaded = 0;
	  } 
	  
		/***** Read config params **********/
	  if(serialState == 'R') {
			final int frame_size_conf = 43;
	    if(serialReadBuf(frame_size_conf+1)) {
	    	if(inBuf[frame_size_conf] == 'R') { // check errors
		    	error = null;
		    	p = 0;
				  // Board configuration (4)
				  version = read8();
				  subversion = read8();
				  int debug = read8();
				  NUM_AXIS = read8(); NUM_AXIS = constrain(NUM_AXIS, 2, 3);
				  
				  
				  // PID data  (18)
				  for(int i=0;i<3;i++) {
				  	confP[i].setValue(read8());
				  	confI[i].setValue(read8()/100.0);
				  	confD[i].setValue(read8());
				  	confPower[i].setValue(read8());
				  	confInv[i].setState(read8() > 0);
				  	confPoles[i].setValue(read8());
				  }

				  // RC data (18)
				  for(int i=0;i<3;i++) {
				  	confRcMinAngle[i].setValue(read16());
				  	confRcMaxAngle[i].setValue(read16());
				  	confRcLPF[i].setValue(read8());
				  	confRcInc[i].setState(read8()>0);
				  }
				  
				  // Ext FC data (2)
				  for(int i=0;i<2;i++) {
				  	confExtGain[i].setValue((byte)read8());
				  }
				  
				  // Other tuning (1)
				  confGyroLPF.setValue(read8());
			
				  
				  /* Enable write button */
				  buttonWRITE.setColorBackground(green_);
				  writeEnable = true;  
					
					/* Show/hide controls */
					showDebug = debug > 0;
					showControls(DEBUG_CONTROLS, showDebug);
					showControls(YAW_CONTROLS, NUM_AXIS >= 3);
				} else {
			    error = getErrorInfo(ERR_GUI_VERSION);
			  }
			  
			  serialState = 0;
			}
		}
	
	  /********** Read debug data *********/
	  else if (serialState == 'D') {
			final int frame_size_debug = 47;
		
	    if(serialReadBuf(frame_size_debug+1)) {
	    	if(inBuf[frame_size_debug] == 'D') { // check errors
		    	p = 0;
			    int ax = read16(), ay = read16(), az = read16();
			    int gx = read16()/10, gy = read16()/10, gz = read16()/10;             //12
			    int magx = read16(), magy = read16(), magz = read16();      
			    int debug1 = read16(), debug2 = read16(), debug3 = read16(), debug4 = read16();  // 26
			    int rcRoll = read16(), rcPitch = read16(), rcYaw = read16();      // 32
			    int extRoll =  read16(), extPitch = read16();  // 36
			    angx = read16()/10.0f; angy = read16()/10.0f;    
			    head = read16()/10.0f;  
			    cycleTime = read16();
			    i2cError = read16();   // 46
			    error = getErrorInfo(read8());
			    
			    avgRoll.addData(angx); avgPitch.addData(angy);
			    peakRoll.addData(angx - avgRoll.res()); peakPitch.addData(angy - avgPitch.res());
			      
			      
			    accROLL.addVal(ax);accPITCH.addVal(ay);accYAW.addVal(az);gyroROLL.addVal(gx);gyroPITCH.addVal(gy);gyroYAW.addVal(gz);
			    magxData.addVal(magx);magyData.addVal(magy);magzData.addVal(magz);
			    debug1Data.addVal(debug1);debug2Data.addVal(debug2);debug3Data.addVal(debug3);debug4Data.addVal(debug4);
		
				  axSlider.setValue(ax);aySlider.setValue(ay);azSlider.setValue(az);gxSlider.setValue(gx);gySlider.setValue(gy);gzSlider.setValue(gz);
				  magxSlider.setValue(magx);magySlider.setValue(magy);magzSlider.setValue(magz);
				  debug1Slider.setValue(debug1);debug2Slider.setValue(debug2);debug3Slider.setValue(debug3);debug4Slider.setValue(debug4);
				
				  rcStickRollSlider.setValue(rcRoll);rcStickPitchSlider.setValue(rcPitch);rcStickYawSlider.setValue(rcYaw);
				  extRollSlider.setValue(extRoll); extPitchSlider.setValue(extPitch);
				  
				  // Show error hints
				  if(error != null) {
			  		buttonCALIBRATE_ACC.setColorBackground(error.code == ERR_CALIB_ACC && timeToFlash() ? red_ : green_);
			  		confPower[0].setColorBackground(error.code == ERR_SET_POWER && confPower[0].value() == 0 && timeToFlash() ? red_ : gray_);
			  		confPower[1].setColorBackground(error.code == ERR_SET_POWER && confPower[1].value() == 0 && timeToFlash() ? red_ : gray_);
			  		confPower[2].setColorBackground(error.code == ERR_SET_POWER && confPower[2].value() == 0 && timeToFlash() ? red_ : gray_);
			  		buttonAUTO_POLES.setColorBackground(error.code == ERR_CALIB_POLES && timeToFlash() ? red_ : blue_);
				  }
				  		
				} else {
			    error = getErrorInfo(ERR_GUI_VERSION);
			  }
	
			  serialState = 0;
		  }
	  } 
	  
	  /***** Read debug vars **********/
	  else if(serialState == 'r') {
	    final int frame_size_conf = DEBUG_VARS_NUM*2;
	    if(serialReadBuf(frame_size_conf+1)) {
	    	if(inBuf[frame_size_conf] == 'r') { // check errors
		    	p = 0;
				  for(int i=0;i<DEBUG_VARS_NUM;i++) {
				  	confDebug[i].setValue(read16());
				  }
				} else {
			    error = getErrorInfo(ERR_GUI_VERSION);
			  }
			}
		  serialState = 0;
		}
		
	  /********** Received serial ESC commands, skip it *********/
	  else if (serialState == 'P' || serialState == 'A') { 
	  	g_serial.read();
	  	serialState = 0;
	  }
	  

	  else {
			serialState = 0;
	    error = getErrorInfo(ERR_GUI_VERSION);
	    g_serial.readBytesUntil('D', inBuf);
	  }
	}
}


//********************************************************
//********************************************************
//********************************************************

class cDataArray {
  float[] m_data;
  int m_maxSize, m_startIndex = 0, m_endIndex = 0, m_curSize;
  
  cDataArray(int maxSize){
    m_maxSize = maxSize;
    m_data = new float[maxSize];
  }
  void addVal(float val) {
    m_data[m_endIndex] = val;
    m_endIndex = (m_endIndex+1)%m_maxSize;
    if (m_curSize == m_maxSize) {
      m_startIndex = (m_startIndex+1)%m_maxSize;
    } else {
      m_curSize++;
    }
  }
  float getVal(int index) {return m_data[(m_startIndex+index)%m_maxSize];}
  int getCurSize(){return m_curSize;}
  int getMaxSize() {return m_maxSize;}
  float getMaxVal() {
    float res = 0.0;
    for(int i=0; i<m_curSize-1; i++) if ((m_data[i] > res) || (i==0)) res = m_data[i];
    return res;
  }
  float getMinVal() {
    float res = 0.0;
    for(int i=0; i<m_curSize-1; i++) if ((m_data[i] < res) || (i==0)) res = m_data[i];
    return res;
  }
  float getRange() {return getMaxVal() - getMinVal();}
}

// This class takes the data and helps graph it
class cGraph {
  float m_gWidth, m_gHeight, m_gLeft, m_gBottom, m_gRight, m_gTop;
  
  cGraph(float x, float y, float w, float h) {
    m_gWidth     = w; m_gHeight    = h;
    m_gLeft      = x; m_gBottom    = y;
    m_gRight     = x + w;
    m_gTop       = y + h;
  }
  
  void drawGraphBox() {
    stroke(0, 0, 0);
    rectMode(CORNERS);
    rect(m_gLeft, m_gBottom, m_gRight, m_gTop);
  }
  
  void drawLine(cDataArray data, float minRange, float maxRange) {
    float graphMultX = m_gWidth/data.getMaxSize();
    float graphMultY = m_gHeight/(maxRange-minRange);
    
    for(int i=0; i<data.getCurSize()-1; ++i) {
      float x0 = i*graphMultX+m_gLeft;
      float y0 = m_gTop-(((data.getVal(i)-(maxRange+minRange)/2)*scaleSlider.value()+(maxRange-minRange)/2)*graphMultY);
      float x1 = (i+1)*graphMultX+m_gLeft;
      float y1 = m_gTop-(((data.getVal(i+1)-(maxRange+minRange)/2 )*scaleSlider.value()+(maxRange-minRange)/2)*graphMultY);
      line(x0, y0, x1, y1);
    }
  }
}



/* Exponential moving average filter */
class AvgFilter {
	private float res = 0;
	private float factor, inv_factor;
	
	AvgFilter(float factor) {
		this.factor = factor;
		this.inv_factor = 1.0f/(factor + 1.0f);
	}
	
	public void addData(float val) {
		res = (res*factor + val) * inv_factor;
	}
	
	public float res() {
		return res;
	}
}


/* Peak meter: detects min and max in the buffered data queue */
class PeakMeter {
	private float buf[];
	private int pos = 0;
	private float min = 0;
	private float max = 0;

	PeakMeter(int size) {
		buf = new float[size];
	}
	
	public void addData(float data) {
		buf[pos] = data;
		pos++; pos%= buf.length;
		
		min = buf[pos];
		max = buf[pos];
		for(int i=0; i<buf.length; i++) {
			int j = (i + pos) % buf.length;
			if(buf[j] > max) max = buf[j];
			if(buf[j] < min) min = buf[j];
		}
	}
	
	public float getMin() {
		return min;
	}
	
	public float getMax() {
		return max;
	}
}

// Change value each ~0.5 sec
public boolean timeToFlash() {
	return ((cycleCnt&(1<<3)) != 0);
}




// Debug controlers names
public static final String DEBUG_CONTROLS[] = { "DEBUG1", "DEBUG2", "DEBUG3", "DEBUG4",
	"debug1", "debug2",  "debug3", "debug4",
	"debug1Slider", "debug2Slider", "debug3Slider", "debug4Slider", 
	"debugVars", 
	"labelVar0", "labelVar1", "labelVar2", "labelVar3", "labelVar4",
	"confDebug0", "confDebug1", "confDebug2", "confDebug3", "confDebug4",
	"SAVE_DEBUG"
	
};
public boolean showDebug = false;

// YAW axis controllers names
public static final String YAW_CONTROLS[] = { 
	"rcYaw", "rangeLabel2", "confRcMinAngle2", "confRcMaxAngle2", "confRcLPF2", "confRcInc2", 
	"confLabel2", "confP2", "confI2", "confD2", "confPower2", "confInv2", "confPoles2"
};

// Hide or show  controlers
public void showControls(final String controls[], boolean show) {

  for(int i=0; i<controls.length; i++) {
  	Controller c = controlP5.controller(controls[i]);
  	if(c != null) {
	  	if(show) { 
				c.show();
			} else {
				c.hide();
			}
		}
  }
}
	
	
